/*
 * Camera test application
 *
 * Copyright (C) 2016 Renesas Electronics Corporation
 * Copyright (C) 2016 Cogent Embedded, Inc. <source@cogentembedded.com>
 *
 * based on:
 *  V4L2 video capture example
 *  This program is provided with the V4L2 API
 *  see http://linuxtv.org/docs.php for more information
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <getopt.h>             /* getopt_long() */

#include <fcntl.h>              /* low-level i/o */
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/ioctl.h>

#include <linux/videodev2.h>
#include <linux/fb.h>

#define CLEAR(x) memset(&(x), 0, sizeof(x))

//#define FIELD V4L2_FIELD_INTERLACED
#define FIELD V4L2_FIELD_NONE

enum io_method {
        IO_METHOD_READ,
        IO_METHOD_MMAP,
        IO_METHOD_USERPTR,
};

struct buffer {
        void   *start;
        size_t  length;
};

static char            *dev_name;
static char            *fbdev_name;
static enum io_method   io = IO_METHOD_MMAP;
//static enum io_method   io = IO_METHOD_USERPTR;
static int              fd = -1;
static int              fbfd = -1;
struct buffer          *buffers;
static unsigned int     n_buffers;
static int              out_buf, out_fb;
static char             *format_name;
static int              frame_count = 70;
static int              fps_count = 0;
static int              framerate = 0;
static int              WIDTH = 1280;
static int              HEIGHT = 800;
static struct fb_var_screeninfo vinfo;
static struct fb_fix_screeninfo finfo;
static long int screensize = 0;
static char *fbmem = 0;

static void errno_exit(const char *s)
{
        fprintf(stderr, "%s error %d, %s\n", s, errno, strerror(errno));
        exit(EXIT_FAILURE);
}

static int xioctl(int fh, int request, void *arg)
{
        int r;

        do {
                r = ioctl(fh, request, arg);
        } while (-1 == r && EINTR == errno);

        return r;
}

#if 0
static inline void yuv_to_rgb32(unsigned char y,
                                unsigned char u,
                                unsigned char v,
                                unsigned char *rgb)
{
    register int r,g,b;

    r = (1192 * (y - 16) + 1634 * (v - 128) ) >> 10;
    g = (1192 * (y - 16) - 833 * (v - 128) - 400 * (u -128) ) >> 10;
    b = (1192 * (y - 16) + 2066 * (u - 128) ) >> 10;

    r = r > 255 ? 255 : r < 0 ? 0 : r;
    g = g > 255 ? 255 : g < 0 ? 0 : g;
    b = b > 255 ? 255 : b < 0 ? 0 : b;

    *rgb++ = b; //B
    *rgb++ = g; //G
    *rgb++ = r; //R
    *rgb   = 0; //A
}
#endif

static void process_image(const void *p, int size)
{
        if (out_buf)
                fwrite(p, size, 1, stdout);

#if 0
        /* for UYVY from camera */
        if (out_fb) {
                unsigned int i, j;
                unsigned char *fbp = (unsigned char *)fbmem;
                char *buf = (char *)p;
                unsigned char Y1, Y2, U, V;

                // assume bpp = 32
                for (i = 0; i < HEIGHT; i++) {
                        for (j = 0; j < WIDTH*2; j+=4) {
                                U = buf[j + 0];
                                Y1 =  buf[j + 1];
                                V = buf[j + 2];
                                Y2 =  buf[j + 3];

                                yuv_to_rgb32(Y1, U, V, &fbp[2*j]);
                                yuv_to_rgb32(Y2, U, V, &fbp[2*(j + 2)]);
                        }

                        fbp += finfo.line_length;
                        buf += (WIDTH*2);
                }
        }
#else
        /* for RGB32 from camera */
        if (out_fb) {
                int i;
                unsigned char *fbp = (unsigned char *)fbmem;
                char *buf = (char *)p;

                for (i = 0; i < HEIGHT; i++) {
                        memcpy(fbp, buf, WIDTH*4);
                        fbp += finfo.line_length;
                        buf += (WIDTH*4);
                }
        }
#endif

//        fflush(stderr);
//        fprintf(stderr, ".");
//        fflush(stdout);
}

static int read_frame(void)
{
        struct v4l2_buffer buf;
        unsigned int i;

        switch (io) {
        case IO_METHOD_READ:
                if (-1 == read(fd, buffers[0].start, buffers[0].length)) {
                        switch (errno) {
                        case EAGAIN:
                                return 0;

                        case EIO:
                                /* Could ignore EIO, see spec. */

                                /* fall through */

                        default:
                                errno_exit("read");
                        }
                }

                process_image(buffers[0].start, buffers[0].length);
                break;

        case IO_METHOD_MMAP:
                CLEAR(buf);

                buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                buf.memory = V4L2_MEMORY_MMAP;

                if (-1 == xioctl(fd, VIDIOC_DQBUF, &buf)) {
                        switch (errno) {
                        case EAGAIN:
                                return 0;

                        case EIO:
                                /* Could ignore EIO, see spec. */

                                /* fall through */

                        default:
                                errno_exit("VIDIOC_DQBUF");
                        }
                }

                assert(buf.index < n_buffers);

                process_image(buffers[buf.index].start, buf.bytesused);

                if (-1 == xioctl(fd, VIDIOC_QBUF, &buf))
                        errno_exit("VIDIOC_QBUF");
                break;

        case IO_METHOD_USERPTR:
                CLEAR(buf);

                buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                buf.memory = V4L2_MEMORY_USERPTR;

                if (-1 == xioctl(fd, VIDIOC_DQBUF, &buf)) {
                        switch (errno) {
                        case EAGAIN:
                                return 0;

                        case EIO:
                                /* Could ignore EIO, see spec. */

                                /* fall through */

                        default:
                                errno_exit("VIDIOC_DQBUF");
                        }
                }

                for (i = 0; i < n_buffers; ++i)
                        if (buf.m.userptr == (unsigned long)buffers[i].start
                            && buf.length == buffers[i].length)
                                break;

                assert(i < n_buffers);

                process_image((void *)buf.m.userptr, buf.bytesused);

                if (-1 == xioctl(fd, VIDIOC_QBUF, &buf))
                        errno_exit("VIDIOC_QBUF");
                break;
        }

        return 1;
}

static inline unsigned long uSecElapsed(struct timeval *t2, struct timeval *t1)
{
    return (t2->tv_sec - t1->tv_sec) * 1000000 + t2->tv_usec - t1->tv_usec;
}

static void fpsCount()
{
    static unsigned frames;
    static struct timeval frame_time;
    static unsigned long usec;
    struct timeval t;

    gettimeofday(&t, NULL);
    usec += frames++ ? uSecElapsed(&t, &frame_time) : 0;
    frame_time = t;
    if (usec >= 1000000) {
        unsigned fps = ((unsigned long long)frames * 10000000 + usec - 1) / usec;
        fprintf(stderr, "%s FPS: %3u.%1u\n", dev_name, fps / 10, fps % 10);
        usec = 0;
        frames = 0;
    }
}

static void mainloop(void)
{
        unsigned int count;

        count = frame_count;

        while (count-- > 0) {
                for (;;) {
                        fd_set fds;
                        struct timeval tv;
                        int r;

                        FD_ZERO(&fds);
                        FD_SET(fd, &fds);

                        /* Timeout. */
                        tv.tv_sec = 2;
                        tv.tv_usec = 0;

                        r = select(fd + 1, &fds, NULL, NULL, &tv);

                        if (-1 == r) {
                                if (EINTR == errno)
                                        continue;
                                errno_exit("select");
                        }

                        if (0 == r) {
                                fprintf(stderr, "select timeout\n");
                                exit(EXIT_FAILURE);
                        }

                        if (read_frame())
                                break;
                        /* EAGAIN - continue select loop. */
                }

                if (fps_count)
                            fpsCount();
        }
}

static void stop_capturing(void)
{
        enum v4l2_buf_type type;

        switch (io) {
        case IO_METHOD_READ:
                /* Nothing to do. */
                break;

        case IO_METHOD_MMAP:
        case IO_METHOD_USERPTR:
                type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                if (-1 == xioctl(fd, VIDIOC_STREAMOFF, &type))
                        errno_exit("VIDIOC_STREAMOFF");
                break;
        }
}

static void start_capturing(void)
{
        unsigned int i;
        enum v4l2_buf_type type;

        switch (io) {
        case IO_METHOD_READ:
                /* Nothing to do. */
                break;

        case IO_METHOD_MMAP:
                for (i = 0; i < n_buffers; ++i) {
                        struct v4l2_buffer buf;

                        CLEAR(buf);
                        buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                        buf.memory = V4L2_MEMORY_MMAP;
                        buf.index = i;

                        if (-1 == xioctl(fd, VIDIOC_QBUF, &buf))
                                errno_exit("VIDIOC_QBUF");
                }
                type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                if (-1 == xioctl(fd, VIDIOC_STREAMON, &type))
                        errno_exit("VIDIOC_STREAMON");
                break;

        case IO_METHOD_USERPTR:
                for (i = 0; i < n_buffers; ++i) {
                        struct v4l2_buffer buf;

                        CLEAR(buf);
                        buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                        buf.memory = V4L2_MEMORY_USERPTR;
                        buf.index = i;
                        buf.m.userptr = (unsigned long)buffers[i].start;
                        buf.length = buffers[i].length;

                        if (-1 == xioctl(fd, VIDIOC_QBUF, &buf))
                                errno_exit("VIDIOC_QBUF");
                }
                type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                if (-1 == xioctl(fd, VIDIOC_STREAMON, &type))
                        errno_exit("VIDIOC_STREAMON");
                break;
        }
}

static void uninit_device(void)
{
        unsigned int i;

        switch (io) {
        case IO_METHOD_READ:
                free(buffers[0].start);
                break;

        case IO_METHOD_MMAP:
                for (i = 0; i < n_buffers; ++i)
                        if (-1 == munmap(buffers[i].start, buffers[i].length))
                                errno_exit("munmap");
                break;

        case IO_METHOD_USERPTR:
                for (i = 0; i < n_buffers; ++i)
                        free(buffers[i].start);
                break;
        }

        free(buffers);
}

static void init_read(unsigned int buffer_size)
{
        buffers = calloc(1, sizeof(*buffers));

        if (!buffers) {
                fprintf(stderr, "Out of memory\n");
                exit(EXIT_FAILURE);
        }

        buffers[0].length = buffer_size;
        buffers[0].start = malloc(buffer_size);

        if (!buffers[0].start) {
                fprintf(stderr, "Out of memory\n");
                exit(EXIT_FAILURE);
        }
}

static void init_mmap(void)
{
        struct v4l2_requestbuffers req;

        CLEAR(req);

        req.count = 4;
        req.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        req.memory = V4L2_MEMORY_MMAP;

        if (-1 == xioctl(fd, VIDIOC_REQBUFS, &req)) {
                if (EINVAL == errno) {
                        fprintf(stderr, "%s does not support "
                                 "memory mapping\n", dev_name);
                        exit(EXIT_FAILURE);
                } else {
                        errno_exit("VIDIOC_REQBUFS");
                }
        }

        if (req.count < 2) {
                fprintf(stderr, "Insufficient buffer memory on %s\n",
                         dev_name);
                exit(EXIT_FAILURE);
        }

        buffers = calloc(req.count, sizeof(*buffers));

        if (!buffers) {
                fprintf(stderr, "Out of memory\n");
                exit(EXIT_FAILURE);
        }

        for (n_buffers = 0; n_buffers < req.count; ++n_buffers) {
                struct v4l2_buffer buf;

                CLEAR(buf);

                buf.type        = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                buf.memory      = V4L2_MEMORY_MMAP;
                buf.index       = n_buffers;

                if (-1 == xioctl(fd, VIDIOC_QUERYBUF, &buf))
                        errno_exit("VIDIOC_QUERYBUF");

                buffers[n_buffers].length = buf.length;
                buffers[n_buffers].start =
                        mmap(NULL /* start anywhere */,
                              buf.length,
                              PROT_READ | PROT_WRITE /* required */,
                              MAP_SHARED /* recommended */,
                              fd, buf.m.offset);

                if (MAP_FAILED == buffers[n_buffers].start)
                        errno_exit("mmap");
        }
}

static void init_userp(unsigned int buffer_size)
{
        struct v4l2_requestbuffers req;

        CLEAR(req);

        req.count  = 4;
        req.type   = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        req.memory = V4L2_MEMORY_USERPTR;

        if (-1 == xioctl(fd, VIDIOC_REQBUFS, &req)) {
                if (EINVAL == errno) {
                        fprintf(stderr, "%s does not support "
                                 "user pointer i/o\n", dev_name);
                        exit(EXIT_FAILURE);
                } else {
                        errno_exit("VIDIOC_REQBUFS");
                }
        }

        buffers = calloc(4, sizeof(*buffers));

        if (!buffers) {
                fprintf(stderr, "Out of memory\n");
                exit(EXIT_FAILURE);
        }

        for (n_buffers = 0; n_buffers < 4; ++n_buffers) {
                buffers[n_buffers].length = buffer_size;
                buffers[n_buffers].start = malloc(buffer_size);

                if (!buffers[n_buffers].start) {
                        fprintf(stderr, "Out of memory\n");
                        exit(EXIT_FAILURE);
                }
        }
}

static void init_device(void)
{
        struct v4l2_capability cap;
        struct v4l2_cropcap cropcap;
        struct v4l2_crop crop;
        struct v4l2_format fmt;
        unsigned int min;

        if (-1 == xioctl(fd, VIDIOC_QUERYCAP, &cap)) {
                if (EINVAL == errno) {
                        fprintf(stderr, "%s is no V4L2 device\n",
                                 dev_name);
                        exit(EXIT_FAILURE);
                } else {
                        errno_exit("VIDIOC_QUERYCAP");
                }
        }

        if (!(cap.capabilities & V4L2_CAP_VIDEO_CAPTURE)) {
                fprintf(stderr, "%s is no video capture device\n",
                         dev_name);
                exit(EXIT_FAILURE);
        }

        switch (io) {
        case IO_METHOD_READ:
                if (!(cap.capabilities & V4L2_CAP_READWRITE)) {
                        fprintf(stderr, "%s does not support read i/o\n",
                                 dev_name);
                        exit(EXIT_FAILURE);
                }
                break;

        case IO_METHOD_MMAP:
        case IO_METHOD_USERPTR:
                if (!(cap.capabilities & V4L2_CAP_STREAMING)) {
                        fprintf(stderr, "%s does not support streaming i/o\n",
                                 dev_name);
                        exit(EXIT_FAILURE);
                }
                break;
        }


        /* Select video input, video standard and tune here. */
        CLEAR(cropcap);
        cropcap.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        if (0 == xioctl(fd, VIDIOC_CROPCAP, &cropcap)) {
                crop.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                crop.c = cropcap.defrect; /* reset to default */

                if (-1 == xioctl(fd, VIDIOC_S_CROP, &crop)) {
                        switch (errno) {
                        case EINVAL:
                                /* Cropping not supported. */
                                break;
                        default:
                                /* Errors ignored. */
                                break;
                        }
                }
        } else {
                /* Errors ignored. */
        }

        if (framerate) {
            struct v4l2_streamparm parm;

            parm.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
            if (-1 == xioctl(fd, VIDIOC_G_PARM, &parm))
                errno_exit("VIDIOC_G_PARM");

            parm.parm.capture.timeperframe.numerator = 1;
            parm.parm.capture.timeperframe.denominator = framerate;
            if (-1 == xioctl(fd, VIDIOC_S_PARM, &parm))
                errno_exit("VIDIOC_S_PARM");
        }

#if 0
    struct v4l2_control control;

    memset(&control, 0, sizeof (control));
    control.id = V4L2_CID_BRIGHTNESS;
//    control.id = V4L2_CID_VFLIP;
//    control.id = V4L2_CID_CONTRAST;
//    control.id = V4L2_CID_SATURATION;
//    control.id = V4L2_CID_GAMMA;
//    control.id = V4L2_CID_GAIN;
//    control.id = V4L2_CID_EXPOSURE;
//    control.id = V4L2_CID_AUTOGAIN;

    if (-1 == xioctl(fd, VIDIOC_G_CTRL, &control))
        errno_exit("VIDIOC_G_CTRL");

    control.value = 0xf0;

    if (-1 == ioctl(fd, VIDIOC_S_CTRL, &control))
        errno_exit("VIDIOC_S_CTRL");
#endif

        CLEAR(fmt);

        fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        fmt.fmt.pix.width  = WIDTH;
        fmt.fmt.pix.height = HEIGHT;
        fmt.fmt.pix.field  = FIELD;

        if (!strncmp(format_name, "uyvy", 4))
                fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_UYVY;
        else if (!strncmp(format_name, "yuyv", 4))
                fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUYV;
        else if (!strncmp(format_name, "rgb565", 5))
                fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB565;
        else if (!strncmp(format_name, "rgb32", 5))
                fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_XBGR32;
        else {
                /* Preserve original settings as set by v4l2-ctl for example */
                if (-1 == xioctl(fd, VIDIOC_G_FMT, &fmt))
                        errno_exit("VIDIOC_G_FMT");
        }

        if (-1 == xioctl(fd, VIDIOC_S_FMT, &fmt))
                errno_exit("VIDIOC_S_FMT");

        /* Buggy driver paranoia. */
        min = fmt.fmt.pix.width * 2;
        if (fmt.fmt.pix.bytesperline < min)
                fmt.fmt.pix.bytesperline = min;
        min = fmt.fmt.pix.bytesperline * fmt.fmt.pix.height;
        if (fmt.fmt.pix.sizeimage < min)
                fmt.fmt.pix.sizeimage = min;

        switch (io) {
        case IO_METHOD_READ:
                init_read(fmt.fmt.pix.sizeimage);
                break;

        case IO_METHOD_MMAP:
                init_mmap();
                break;

        case IO_METHOD_USERPTR:
                init_userp(fmt.fmt.pix.sizeimage);
                break;
        }
}

static void close_device(void)
{
        if (out_fb) {
                munmap(fbmem, screensize);
                close(fbfd);
                fbfd = -1;
        }

        if (-1 == close(fd))
                errno_exit("close");

        fd = -1;
}

static void open_device(void)
{
        struct stat st;

        if (-1 == stat(dev_name, &st)) {
                fprintf(stderr, "Cannot identify '%s': %d, %s\n",
                         dev_name, errno, strerror(errno));
                exit(EXIT_FAILURE);
        }

        if (!S_ISCHR(st.st_mode)) {
                fprintf(stderr, "%s is no device\n", dev_name);
                exit(EXIT_FAILURE);
        }

        fd = open(dev_name, O_RDWR /* required */ | O_NONBLOCK, 0);

        if (-1 == fd) {
                fprintf(stderr, "Cannot open '%s': %d, %s\n",
                         dev_name, errno, strerror(errno));
                exit(EXIT_FAILURE);
        }

        if (out_fb) {
                fbfd = open(fbdev_name, O_RDWR, 0);
                if (-1 == fbfd) {
                        fprintf(stderr, "Cannot open '%s': %d, %s\n",
                                 fbdev_name, errno, strerror(errno));
                        exit(EXIT_FAILURE);
                }

                // Get fixed screen information
                if (ioctl(fbfd, FBIOGET_FSCREENINFO, &finfo) == -1) {
                        perror("Error reading fixed information");
                        exit(EXIT_FAILURE);
                }
                // Get variable screen information
                if (ioctl(fbfd, FBIOGET_VSCREENINFO, &vinfo) == -1) {
                        perror("Error reading variable information");
                        exit(EXIT_FAILURE);
                }
//                printf("%dx%d, %dbpp\n", vinfo.xres, vinfo.yres, vinfo.bits_per_pixel);

                printf("fb0 Fixed Info:\n"
                                "   %s  @ 0x%lx, len=%d, line=%d bytes,\n",
                        finfo.id,
                        finfo.smem_start,
                        finfo.smem_len,
                        finfo.line_length);

                printf("   Geometry - %u x %u, %u bpp%s\n",
                        vinfo.xres,
                        vinfo.yres,
                        vinfo.bits_per_pixel,
                        vinfo.grayscale ? ", greyscale" : "");


                // Figure out the size of the screen in bytes
                screensize = finfo.smem_len; //vinfo.xres * vinfo.yres * vinfo.bits_per_pixel / 8;

                // Map the device to memory
                fbmem = (char *)mmap(0, screensize, PROT_READ | PROT_WRITE, MAP_SHARED, fbfd, 0);
                if ((int)fbmem == -1) {
                        perror("Error: failed to map framebuffer device to memory");
                        exit(EXIT_FAILURE);
                }
        }
}

static void usage(FILE *fp, char **argv)
{
        fprintf(fp,
                 "Usage: %s [options]\n\n"
                 "Version 1.3\n"
                 "Options:\n"
                 "-d | --device name   Video device name [%s]\n"
                 "-h | --help          Print this message\n"
                 "-m | --mmap          Use memory mapped buffers [default]\n"
                 "-r | --read          Use read() calls\n"
                 "-u | --userp         Use application allocated buffers\n"
                 "-o | --output        Outputs stream to stdout\n"
                 "-F | --output_fb     Outputs stream to framebuffer\n"
                 "-f | --format        Set pixel format: uyvy, yuyv, rgb565, rgb32 [%s]\n"
                 "-c | --count         Number of frames to grab [%i]\n"
                 "-z | --fps_count     Enable fps show\n"
                 "-s | --framerate     Set framerate\n"
                 "-W | --width         Video width [%i]\n"
                 "-H | --height        Video height [%i]\n"
                 "",
                 argv[0], dev_name, format_name, frame_count, WIDTH, HEIGHT);
}

static const char short_options[] = "d:hmruoFf:c:zs:W:H:";

static const struct option
long_options[] = {
        { "device", required_argument, NULL, 'd' },
        { "help",   no_argument,       NULL, 'h' },
        { "mmap",   no_argument,       NULL, 'm' },
        { "read",   no_argument,       NULL, 'r' },
        { "userp",  no_argument,       NULL, 'u' },
        { "output", no_argument,       NULL, 'o' },
        { "output_fb", no_argument,    NULL, 'F' },
        { "format", required_argument, NULL, 'f' },
        { "count",  required_argument, NULL, 'c' },
        { "fps_count",  required_argument, NULL, 'z' },
        { "framerate",  required_argument, NULL, 's' },
        { "width",  required_argument, NULL, 'W' },
        { "height",  required_argument, NULL, 'H' },
        { 0, 0, 0, 0 }
};

int main(int argc, char **argv)
{
        dev_name = "/dev/video0";
        fbdev_name = "/dev/fb0";
        format_name = "uyvy";

        for (;;) {
                int idx;
                int c;

                c = getopt_long(argc, argv,
                                short_options, long_options, &idx);

                if (-1 == c)
                        break;

                switch (c) {
                case 0: /* getopt_long() flag */
                        break;

                case 'd':
                        dev_name = optarg;
                        break;

                case 'h':
                        usage(stdout, argv);
                        exit(EXIT_SUCCESS);

                case 'm':
                        io = IO_METHOD_MMAP;
                        break;

                case 'r':
                        io = IO_METHOD_READ;
                        break;

                case 'u':
                        io = IO_METHOD_USERPTR;
                        break;

                case 'o':
                        out_buf++;
                        break;

                case 'F':
                        out_fb++;
                        break;

                case 'f':
                        format_name = optarg;
                        break;

                case 'c':
                        errno = 0;
                        frame_count = strtol(optarg, NULL, 0);
                        if (errno)
                                errno_exit(optarg);
                        break;

                case 'z':
                        fps_count = 1;
                        break;

                case 's':
                        errno = 0;
                        framerate = strtol(optarg, NULL, 0);
                        if (errno)
                                errno_exit(optarg);
                        break;

                case 'W':
                        errno = 0;
                        WIDTH = strtol(optarg, NULL, 0);
                        if (errno)
                                errno_exit(optarg);
                        break;

                case 'H':
                        errno = 0;
                        HEIGHT = strtol(optarg, NULL, 0);
                        if (errno)
                                errno_exit(optarg);
                        break;

                default:
                        usage(stderr, argv);
                        exit(EXIT_FAILURE);
                }
        }

        open_device();
        init_device();
        start_capturing();
        mainloop();
        stop_capturing();
        uninit_device();
        close_device();
//        fprintf(stderr, "\n");
        return 0;
}
