/*
 * Read/Write firmware to SPI Flash connected to V4L2 subchips (f.e. ov490)
 *
 * Copyright (C) 2016 Renesas Electronics Europe GmbH
 * Copyright (C) 2016 Renesas Electronics Corporation
 * Copyright (C) 2016 Cogent Embedded, Inc.
 *
 * based on v4l2-dbg.cpp from v4l-utils by Hans Verkuil <hverkuil@xs4all.nl>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Suite 500, Boston, MA  02110-1335  USA
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>

#include <linux/videodev2.h>

static struct v4l2_dbg_register set_reg;
static struct v4l2_dbg_register get_reg;
static struct v4l2_dbg_match match;
static int fd = -1;
static FILE *file;

enum Option {
	OptGetRegister = 'g',
	OptSetRegister = 's',
	OptReadFirmware = 'r',
	OptWriteFirmware = 'w',
	OptSetDevice = 'd',
	OptGetDriverInfo = 'D',
	OptChip = 'c',
	OptScanChips = 'n',
	OptHelp = 'h',

	OptVerbose,
	OptLast = 256
};

static char options[OptLast];

static unsigned capabilities;

static struct option long_options[] = {
	{"device", required_argument, 0, OptSetDevice},
	{"help", no_argument, 0, OptHelp},
	{"get-register", required_argument, 0, OptGetRegister},
	{"set-register", required_argument, 0, OptSetRegister},
	{"read-firmware", required_argument, 0, OptReadFirmware},
	{"write-firmware", required_argument, 0, OptWriteFirmware},
	{"chip", required_argument, 0, OptChip},
	{"scan-chips", no_argument, 0, OptScanChips},
	{"info", no_argument, 0, OptGetDriverInfo},
	{"verbose", no_argument, 0, OptVerbose},
	{0, 0, 0, 0}
};

static void usage(void)
{
	printf("Usage: v4l2-dbg [options] [values]\n"
	       "  -D, --info         Show driver info [VIDIOC_QUERYCAP]\n"
	       "  -d, --device=<dev> Use device <dev> instead of /dev/video0\n"
	       "                     If <dev> starts with a digit, then /dev/video<dev> is used\n"
	       "  -h, --help         Display this help message\n"
	       "  --verbose          Turn on verbose ioctl error reporting\n"
	       "  -c, --chip=<chip>  The chip identifier to use with other commands\n"
	       "                     It can be one of:\n"
	       "                         bridge<num>: bridge chip number <num>\n"
	       "                         bridge (default): same as bridge0\n"
	       "                         subdev<num>: sub-device number <num>\n"
	       "  -g, --get-register=<addr>\n"
	       "		     Get the specified register [VIDIOC_DBG_G_REGISTER]\n"
	       "  -s, --set-register=<addr>\n"
	       "		     Set the register with the commandline arguments\n"
	       "                     The register will autoincrement [VIDIOC_DBG_S_REGISTER]\n"
	       "  -r, --read-firmware=<file>\n"
	       "		     Read the SPI firmware to the file\n"
	       "  -w, --write-firmware=<file>\n"
	       "		     Write SPI firmware from file\n"
	       "  -n, --scan-chips   Scan the available bridge and subdev chips [VIDIOC_DBG_G_CHIP_INFO]\n"
	       );
	exit(0);
}

static void print_name(struct v4l2_dbg_chip_info *chip)
{
	printf("%-10s (%c%c)\n", chip->name,
		(chip->flags & V4L2_CHIP_FL_READABLE) ? 'r' : '-',
		(chip->flags & V4L2_CHIP_FL_WRITABLE) ? 'w' : '-');
}

static const char *binary(unsigned long long val)
{
	static char bin[80];
	char *p = bin;
	int i, j;
	int bits = 64;

	if ((val & 0xffffffff00000000LL) == 0) {
		if ((val & 0xffff0000) == 0) {
			if ((val & 0xff00) == 0)
				bits = 8;
			else
				bits= 16;
		}
		else
			bits = 32;
	}

	for (i = bits - 1; i >= 0; i -= 8) {
		for (j = i; j >= i - 7; j--) {
			if (val & (1LL << j))
				*p++ = '1';
			else
				*p++ = '0';
		}
		*p++ = ' ';
	}
	p[-1] = 0;
	return bin;
}

static int doioctl(int fd, unsigned long int request, void *parm, const char *name)
{
	int retVal = ioctl(fd, request, parm);

	if (options[OptVerbose]) {
		if (retVal < 0)
			printf("%s: failed: %s\n", name, strerror(errno));
		else
			printf("%s: ok\n", name);
	}

	return retVal;
}

static int parse_subopt(char **subs, const char * const *subopts, char **value)
{
	int opt = getsubopt(subs, (char * const *)subopts, value);

	if (opt == -1) {
		fprintf(stderr, "Invalid suboptions specified\n");
		usage();
		exit(1);
	}
	if (*value == NULL) {
		fprintf(stderr, "No value given to suboption <%s>\n",
				subopts[opt]);
		usage();
		exit(1);
	}
	return opt;
}

static int v4l2_read(__u64 reg)
{
	get_reg.match = match;
	get_reg.reg = reg;

	if (ioctl(fd, VIDIOC_DBG_G_REGISTER, &get_reg) < 0) {
		printf("ioctl: VIDIOC_DBG_G_REGISTER "
			"failed for 0x%llx\n", get_reg.reg);
		return -1;
	}

	return get_reg.val;
}

static void v4l2_write(__u64 reg, __u64 val)
{
	set_reg.match = match;
	set_reg.reg = reg;
	set_reg.val = val;

	if (ioctl(fd, VIDIOC_DBG_S_REGISTER, &set_reg) < 0) {
		printf("Failed to set register 0x%08llx value 0x%llx: %s\n",
			set_reg.reg, set_reg.val, strerror(errno));
	}
}

static __u8 v4l2_readb(__u32 reg)
{
	v4l2_write(0xfffd, (reg >> 24) & 0xff);
	v4l2_write(0xfffe, (reg >> 16) & 0xff);
	return v4l2_read(reg & 0xffff);
}

static void v4l2_writeb(__u32 reg, __u8 val)
{
	if (options[OptVerbose])
		printf("writeb 0x%x -> 0x%x\n", val, reg);
	v4l2_write(0xfffd, (reg >> 24) & 0xff);
	v4l2_write(0xfffe, (reg >> 16) & 0xff);
	v4l2_write(reg & 0xffff, val);
}

#define ReadReg8 v4l2_readb
#define WriteReg8 v4l2_writeb
#include "spi_operation.c"

int main(int argc, char **argv)
{
	const char *device = "/dev/video0";	/* -d device */
	char *file_firmware;
	int file_size = 0x8000, n_read;
	struct stat file_st;
	struct v4l2_capability vcap;	/* list_cap */
	struct v4l2_dbg_chip_info chip_info;
	char short_options[26 * 2 * 3 + 1];
	int ch, idx = 0;
	int i;
	char *data;
	int spi_addr;

	match.type = V4L2_CHIP_MATCH_SUBDEV;
	match.addr = 0;
	memset(&set_reg, 0, sizeof(set_reg));
	memset(&get_reg, 0, sizeof(get_reg));

	if (argc == 1) {
		usage();
		return 0;
	}
	for (i = 0; long_options[i].name; i++) {
		if (!isalpha(long_options[i].val))
			continue;
		short_options[idx++] = long_options[i].val;
		if (long_options[i].has_arg == required_argument) {
			short_options[idx++] = ':';
		} else if (long_options[i].has_arg == optional_argument) {
			short_options[idx++] = ':';
			short_options[idx++] = ':';
		}
	}
	while (1) {
		int option_index = 0;

		short_options[idx] = 0;
		ch = getopt_long(argc, argv, short_options,
				 long_options, &option_index);
		if (ch == -1)
			break;

		options[(int)ch] = 1;
		switch (ch) {
		case OptHelp:
			usage();
			return 0;

		case OptSetDevice:
			device = optarg;
			if (device[0] >= '0' && device[0] <= '9' && strlen(device) <= 3) {
				static char newdev[20];

				sprintf(newdev, "/dev/video%s", device);
				device = newdev;
			}
			break;

		case OptChip:
			if (!memcmp(optarg, "subdev", 6) && isdigit(optarg[6])) {
				match.type = V4L2_CHIP_MATCH_SUBDEV;
				match.addr = strtoul(optarg + 6, NULL, 0);
				break;
			}
			if (!memcmp(optarg, "bridge", 6)) {
				match.type = V4L2_CHIP_MATCH_BRIDGE;
				match.addr = strtoul(optarg + 6, NULL, 0);
				break;
			}
			match.type = V4L2_CHIP_MATCH_BRIDGE;
			match.addr = 0;
			break;

		case OptSetRegister:
			if (optind >= argc)
				usage();

			set_reg.reg = strtoull(optarg, NULL, 0);
			set_reg.val = strtoull(argv[optind], NULL, 0);
			break;

		case OptGetRegister:
			get_reg.reg = strtoull(optarg, NULL, 0);
			break;

		case OptReadFirmware:
			file_firmware = optarg;
			if (optind < argc)
				file_size = strtoull(argv[optind], NULL, 0);
			break;

		case OptWriteFirmware:
			file_firmware = optarg;
			break;

		case ':':
			fprintf(stderr, "Option `%s' requires a value\n",
				argv[optind]);
			usage();
			return 1;

		case '?':
			fprintf(stderr, "Unknown argument `%s'\n",
				argv[optind]);
			usage();
			return 1;
		}
	}

	if ((fd = open(device, O_RDWR)) < 0) {
		fprintf(stderr, "Failed to open %s: %s\n", device,
			strerror(errno));
		exit(1);
	}

	doioctl(fd, VIDIOC_QUERYCAP, &vcap, "VIDIOC_QUERYCAP");
	capabilities = vcap.capabilities;

	/* Information Opts */
	if (options[OptGetDriverInfo]) {
		printf("Driver info:\n");
		printf("\tDriver name   : %s\n", vcap.driver);
		printf("\tCard type     : %s\n", vcap.card);
		printf("\tBus info      : %s\n", vcap.bus_info);
		printf("\tDriver version: %d.%d.%d\n",
				vcap.version >> 16,
				(vcap.version >> 8) & 0xff,
				vcap.version & 0xff);
		printf("\tCapabilities  : 0x%08X\n", vcap.capabilities);
	}

	if (options[OptScanChips]) {
		chip_info.match.type = V4L2_CHIP_MATCH_BRIDGE;
		chip_info.match.addr = 0;

		while (doioctl(fd, VIDIOC_DBG_G_CHIP_INFO, &chip_info, "VIDIOC_DBG_G_CHIP_INFO") == 0 && chip_info.name[0]) {
			printf("bridge%d: ", chip_info.match.addr);
			print_name(&chip_info);
			chip_info.match.addr++;
		}

		chip_info.match.type = V4L2_CHIP_MATCH_SUBDEV;
		chip_info.match.addr = 0;
		while (doioctl(fd, VIDIOC_DBG_G_CHIP_INFO, &chip_info, "VIDIOC_DBG_G_CHIP_INFO") == 0 && chip_info.name[0]) {
			printf("subdev%d: ", chip_info.match.addr++);
			print_name(&chip_info);
		}
	}

	if (options[OptGetRegister]) {
		v4l2_read(get_reg.reg);
		printf("Register 0x%08llx", get_reg.reg);
		printf(" = %llxh (%lldd  %sb)\n", get_reg.val, get_reg.val, binary(get_reg.val));
	}

	if (options[OptSetRegister]) {
		v4l2_write(set_reg.reg, set_reg.val);
		printf("Register 0x%08llx", set_reg.reg);
		printf(" set to 0x%llx\n", set_reg.val);
	}

	if (options[OptReadFirmware]) {
		file = fopen(file_firmware, "w");
		if (!file) {
			fprintf(stderr, "Cannot open '%s': %d, %s\n",
				file_firmware , errno, strerror(errno));
			goto exit;
		}

		spi_init();

		file_size = (file_size + g_sf.page_size - 1) & ~(g_sf.page_size - 1);
		data = (char *)malloc(g_sf.page_size);

		spi_addr = 0;
		for (i = 0; i < file_size / g_sf.page_size; i++) {
			spi_page_read(spi_addr, (unsigned int *)data);
			fwrite(data, 1, g_sf.page_size, file);
			spi_addr += g_sf.page_size;
		}

		printf("read firmware size %d to file %s\n", file_size, file_firmware);
		free(data);
		fclose(file);
	}

	if (options[OptWriteFirmware]) {
		file = fopen(file_firmware, "r");
		if (!file) {
			fprintf(stderr, "Cannot open '%s': %d, %s\n",
				file_firmware , errno, strerror(errno));
			goto exit;
		}

		spi_init();

		stat(file_firmware, &file_st);

		file_size = (file_st.st_size + g_sf.page_size - 1) & ~(g_sf.page_size - 1);
		data = (char *)malloc(g_sf.page_size);

		spi_addr = 0;
		spi_erase(spi_addr, ERASE_BLOCK);

		for (i = 0; i < file_size / g_sf.page_size; i++) {
			n_read = fread(data, 1, g_sf.page_size, file);
			if (n_read < g_sf.page_size)
				memset(data + n_read, 0xff, g_sf.page_size - n_read);
			spi_page_write((unsigned int *)data, spi_addr);
			spi_addr += g_sf.page_size;
		}

		printf("written firmware size %d from file %s\n", file_st.st_size, file_firmware);
		free(data);
		fclose(file);
	}

exit:
	close(fd);
	exit(0);
}
