/*
 * Copyright 2018-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *     http://aws.amazon.com/apache2.0/
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

#include "NavigationDispatcher.h"

#include <json-c/json.h>

#include <aasb/Consts.h>

#include "AlexaConsts.h"

namespace agl {
namespace dispatcher {
namespace navigation {

// Shortcut to reach logging level.
using Level = agl::common::interfaces::ILogger::Level;

// Namespaces.
using namespace agl::common::interfaces;
using namespace aasb::bridge;

// Logging Tag for this file.
static std::string TAG = "alexa::plugins::NavigationDispatcher";

// Name of the phone control capability
static std::string VSHL_CAPABILITY_NAVIGATION = "navigation";

// Navigation publish verb to notify vshl capabilities of new event/directive generated by Alexa.
static std::string VSHL_CAPABILITY_VERB_NAVIGATION_PUBLISH = VSHL_CAPABILITY_NAVIGATION + "/publish";

std::shared_ptr<NavigationDispatcher> NavigationDispatcher::create(
    std::shared_ptr<ILogger> logger,
    std::shared_ptr<IAASBController> aasbController,
    std::shared_ptr<IAFBApi> api) {

    return std::shared_ptr<NavigationDispatcher>(new NavigationDispatcher(logger, aasbController, api));
}

NavigationDispatcher::NavigationDispatcher(
    std::shared_ptr<ILogger> logger,
    std::shared_ptr<IAASBController> aasbController,
    std::shared_ptr<IAFBApi> api) :
        m_logger(logger),
        m_aasbController(aasbController),
        m_api(api) {
}

void NavigationDispatcher::onReceivedDirective(
    const std::string& action,
    const std::string& payload) {

    m_logger->log(Level::DEBUG, TAG, "Processing navigation directive: " + action);

    std::string vshlCapabilityAction = action;

    json_object* argsJ = json_object_new_object();
    json_object* actionJ = json_object_new_string(vshlCapabilityAction.c_str());
    json_object* payloadJ = json_object_new_string(payload.c_str());
    json_object_object_add(argsJ, agl::alexa::JSON_ATTR_ACTION.c_str(), actionJ);
    json_object_object_add(argsJ, agl::alexa::JSON_ATTR_PAYLOAD.c_str(), payloadJ);

    auto logger = m_logger;  // To capture it for the lambda.
    m_api->callAsync(
        agl::alexa::VSHL_CAPABILITIES_API_NAME,
        VSHL_CAPABILITY_VERB_NAVIGATION_PUBLISH,
        argsJ,
        [logger](json_object* result, const std::string err, const std::string info) {
            logger->log(Level::DEBUG, TAG, "Navigation publish completed err:" + err);
        });

    m_logger->log(Level::DEBUG, TAG, "Navigation action processing completed");
}

}  // namespace phonecall
}  // namespace dispatcher
}  // namespace agl
