/*
 * Copyright 2017-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *     http://aws.amazon.com/apache2.0/
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.amazon.aace.vehicle.config;

import com.amazon.aace.core.config.EngineConfiguration;

/**
 * The @c VehicleConfiguration class is a factory interface for creating Vehicle service configuration objects.
 */
public class VehicleConfiguration {

    /**
     * Specifies configurable vehicle properties
     */
    public enum VehiclePropertyType {
        /**
         * Vehicle's make
         * @hideinitializer
         */
        MAKE("MAKE","make"),

        /**
         * Vehicle's model
         * @hideinitializer
         */
        MODEL("MODEL","model"),

        /**
         * Vehicle model year, example: 2018, 2019, 2020
         * @hideinitializer
         */
        YEAR("YEAR","year"),

        /**
         * Variations of a model with different set of features, example: Standard, Sport, Limited
         * @hideinitializer
         */
        TRIM("TRIM","trim"),

        /**
         * Current location (country/region/state) of the vehicle, example: US, US-North, WA
         * @hideinitializer
         */
        GEOGRAPHY("GEOGRAPHY","geography"),

        /**
         * OEM client software version
         * @hideinitializer
         */
        VERSION("VERSION","version"),

        /**
         * Operating system for vehicle's infotainment system, example: Android 8.1 Oreo API Level 26
         * @hideinitializer
         */
        OPERATING_SYSTEM("OPERATING_SYSTEM","os"),

        /**
         * Supporting hardware architecture, example: x86_64
         * @hideinitializer
         */
        HARDWARE_ARCH("HARDWARE_ARCH","arch"),

        /**
         * Language or locale selected for Alexa by the vehicle owner, example: en-UA, fr-CA
         * @hideinitializer
         */
        LANGUAGE("LANGUAGE","language"),

        /**
         * Type and arrangement of microphone in the car, example: 7 mic array, centrally mounted
         * @hideinitializer
         */
        MICROPHONE("MICROPHONE","microphone"),

        /**
         * Comma-separated list countries (ISO country codes)
         * @hideinitializer
         */
        COUNTRY_LIST("COUNTRY_LIST","countries"),

        /**
         * Automaker's internal identifier for the vehicle
         * @hideinitializer
         */
        VEHICLE_IDENTIFIER("VEHICLE_IDENTIFIER","vehicleIdentifier");

        /**
         * @internal
         */
        private String mName;

        /**
         * @internal
         */
        private String mKey;

        /**
         * Type used to identify a vehicle property type and value pair
         */
        VehiclePropertyType( String name, String key ) {
            mName = name;
            mKey = key;
        }

        /**
         * @internal
         */
        public String toString() {
            return mName;
        }

        /**
         * @internal
         */
        public String getKey() {
            return mKey;
        }
    }

    public static class VehicleProperty {
        private VehiclePropertyType mType;
        private String mValue;

        public VehicleProperty( VehiclePropertyType type, String value ) {
            mType = type;
            mValue = value;
        }

        public VehiclePropertyType getType() { return mType; }
        public String getValue() { return mValue; }
    }

    /**
     * Factory method used to programmatically generate vehicle info configuration data.
     * The data generated by this method is equivalent to providing the following JSON
     * values in a configuration file:
     *
     * @code{.json}
     * {
     *   "aace.vehicle":
     *   {
     *      "info": {
     *          "make": "<MAKE>",
     *          "model": "<MODEL>",
     *          "year": "<YEAR>",
     *          "trim": "<TRIM>",
     *          "geography": "<GEOGRAPHY>",
     *          "version": "<SOFTWARE_VERSION>",
     *          "os": "<OPERATING_SYSTEM>",
     *          "arch": "<HARDWARE_ARCH>",
     *          "language": "<LANGUAGE>",
     *          "microphone": "<MICROPHONE>",
     *          "countries": "<COUNTRY_LIST>",
     *          "vehicleIdentifier": "<VEHICLE_IDENTIFIER>"
     *      }
     *   }
     * }
     * @endcode
     *
     * @param propertyList A list of @c VehicleProperty type and value pairs
     */
    public static EngineConfiguration createVehicleInfoConfig( final VehicleProperty[] propertyList )
    {
        return new EngineConfiguration() {
            @Override
            protected long createNativeRef() {
                return createVehicleInfoConfigBinder( propertyList );
            }
        };
    }

    // Native Engine JNI methods
    static private native long createVehicleInfoConfigBinder( VehicleProperty[] propertyList );

    /**
     * Factory method used to programmatically generate vehicle operating country configuration data.
     * The data generated by this method is equivalent to providing the following JSON
     * values in a configuration file:
     *
     * @code{.json}
     * {
     *   "aace.vehicle": {
     *      "operatingCountry": "<COUNTRY>"
     *   }
     * }
     * @endcode
     *
     * @param operatingCountry A 2-letter ISO country code
     */
    public static EngineConfiguration createOperatingCountryConfig( final String operatingCountry )
    {
        return new EngineConfiguration() {
            @Override
            protected long createNativeRef() {
                return createOperatingCountryConfigBinder( operatingCountry );
            }
        };
    }

    // Native Engine JNI methods
    static private native long createOperatingCountryConfigBinder( String operatingCountry );
}
