/*
 * Copyright 2017-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *     http://aws.amazon.com/apache2.0/
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

#ifndef AACE_VEHICLE_CONFIG_VEHICLE_CONFIGURATION_H
#define AACE_VEHICLE_CONFIG_VEHICLE_CONFIGURATION_H

#include <utility>

#include "AACE/Core/EngineConfiguration.h"

/** @file */

namespace aace {
namespace vehicle {
namespace config {

/**
 * The @c VehicleConfiguration class is a factory interface for creating Vehicle service configuration objects.
 */
class VehicleConfiguration {
public:

    /**
     * Specifies configurable vehicle properties
     */
    enum class VehiclePropertyType {

        /**
         * Vehicle's make
         */
        MAKE,

        /**
         * Vehicle's model
         */
        MODEL,

        /**
         * Vehicle model year, example: 2018, 2019, 2020
         */
        YEAR,

        /**
         * Variations of a model with different set of features, example: Standard, Sport, Limited
         */
        TRIM,

        /**
         * Current location (country/region/state) of the vehicle, example: US, US-North, WA
         */
        GEOGRAPHY,

        /**
         * OEM client software version
         */
        VERSION,

        /**
         * Operating system for vehicle's infotainment system, example: Android 8.1 Oreo API Level 26
         */
        OPERATING_SYSTEM,

        /**
         * Supporting hardware architecture, example: x86_64
         */
        HARDWARE_ARCH,

        /**
         * Language or locale selected for Alexa by the vehicle owner, example: en-UA, fr-CA
         */
        LANGUAGE,

        /**
         * Type and arrangement of microphone in the car, example: 7 mic array, centrally mounted
         */
        MICROPHONE,
        
        /**
         * Comma-separated list countries (ISO country codes)
         */
        COUNTRY_LIST,

        /**
         * Automaker's internal identifier for the vehicle
         */
        VEHICLE_IDENTIFIER
    };
    
    /**
     * Identifies a vehicle property with a type and value pair
     */
    using VehicleProperty = std::pair<VehiclePropertyType,std::string>;

    /**
     * Factory method used to programmatically generate vehicle info configuration data.
     * The data generated by this method is equivalent to providing the following JSON
     * values in a configuration file:
     *
     * @code{.json}
     * {
     *   "aace.vehicle":
     *   {
     *      "info": {
     *          "make": "<MAKE>",
     *          "model": "<MODEL>",
     *          "year": "<YEAR>",
     *          "trim": "<TRIM>",
     *          "geography": "<GEOGRAPHY>",
     *          "version": "<SOFTWARE_VERSION>",
     *          "os": "<OPERATING_SYSTEM>",
     *          "arch": "<HARDWARE_ARCH>",
     *          "language": "<LANGUAGE>",
     *          "microphone": "<MICROPHONE>",
     *          "countries": "<COUNTRY_LIST>",
     *          "vehicleIdentifier": "<VEHICLE_IDENTIFIER>"
     *      }
     *   }
     * }
     * @endcode
     *
     * @param [in] propertyList A list of @c VehicleProperty type and value pairs
     */
    static std::shared_ptr<aace::core::config::EngineConfiguration> createVehicleInfoConfig( const std::vector<VehicleProperty>& propertyList );
    
    /**
     * Factory method used to programmatically generate vehicle operating country configuration data.
     * The data generated by this method is equivalent to providing the following JSON
     * values in a configuration file:
     *
     * @code{.json}
     * {
     *   "aace.vehicle": {
     *      "operatingCountry": "<COUNTRY>"
     *   }
     * }
     * @endcode
     *
     * @param [in] operatingCountry A 2-letter ISO country code
     */
    static std::shared_ptr<aace::core::config::EngineConfiguration> createOperatingCountryConfig( const std::string& operatingCountry );
};

inline std::ostream& operator<<(std::ostream& stream, const VehicleConfiguration::VehiclePropertyType& state) {
    switch (state) {
        case VehicleConfiguration::VehiclePropertyType::MAKE:
            stream << "MAKE";
            break;
        case VehicleConfiguration::VehiclePropertyType::MODEL:
            stream << "MODEL";
            break;
        case VehicleConfiguration::VehiclePropertyType::YEAR:
            stream << "YEAR";
            break;
        case VehicleConfiguration::VehiclePropertyType::TRIM:
            stream << "TRIM";
            break;
        case VehicleConfiguration::VehiclePropertyType::GEOGRAPHY:
            stream << "GEOGRAPHY";
            break;
        case VehicleConfiguration::VehiclePropertyType::VERSION:
            stream << "VERSION";
            break;
        case VehicleConfiguration::VehiclePropertyType::OPERATING_SYSTEM:
            stream << "OPERATING_SYSTEM";
            break;
        case VehicleConfiguration::VehiclePropertyType::HARDWARE_ARCH:
            stream << "HARDWARE_ARCH";
            break;
        case VehicleConfiguration::VehiclePropertyType::LANGUAGE:
            stream << "LANGUAGE";
            break;
        case VehicleConfiguration::VehiclePropertyType::MICROPHONE:
            stream << "MICROPHONE";
            break;
        case VehicleConfiguration::VehiclePropertyType::COUNTRY_LIST:
            stream << "COUNTRY_LIST";
            break;
        case VehicleConfiguration::VehiclePropertyType::VEHICLE_IDENTIFIER:
            stream << "VEHICLE_IDENTIFIER";
            break;
    }
    return stream;
}
    
} // aace::vehicle::config
} // aace::vehicle
} // aace

#endif // AACE_VEHICLE_CONFIG_VEHICLE_CONFIGURATION_H
