/*
 * Copyright 2018-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *     http://aws.amazon.com/apache2.0/
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

#include "CarControlDispatcher.h"

#include <json-c/json.h>

#include <aasb/Consts.h>

#include "AlexaConsts.h"

namespace agl {
namespace dispatcher {
namespace localVoiceControl {
namespace carControl {

// Shortcut to reach logging level.
using Level = agl::common::interfaces::ILogger::Level;

// Namespaces.
using namespace agl::common::interfaces;
using namespace aasb::bridge;

// Logging Tag for this file.
static std::string TAG = "alexa::plugins::CarControlDispatcher";

// Name of the carcontrol capability
static std::string VSHL_CAPABILITY_CAR_CONTROL = "carcontrol";

// Carcontrol publish verb to notify vshl capabilities of new event/directive generated by Alexa.
static std::string VSHL_CAPABILITY_VERB_CAR_CONTROL_PUBLISH = VSHL_CAPABILITY_CAR_CONTROL + "/publish";

// Carcontrol subscribe verb to subscribe for vshl capabilities events from Apps.
static std::string VSHL_CAPABILITY_VERB_CAR_CONTROL_SUBSCRIBE = VSHL_CAPABILITY_CAR_CONTROL + "/subscribe";

std::shared_ptr<CarControlDispatcher> CarControlDispatcher::create(
    std::shared_ptr<ILogger> logger,
    std::shared_ptr<IAASBController> aasbController,
    std::shared_ptr<IAFBApi> api) {
    return std::shared_ptr<CarControlDispatcher>(new CarControlDispatcher(logger, aasbController, api));
}

CarControlDispatcher::CarControlDispatcher(
    std::shared_ptr<ILogger> logger,
    std::shared_ptr<IAASBController> aasbController,
    std::shared_ptr<IAFBApi> api) :
        m_logger(logger), m_aasbController(aasbController), m_api(api) {
}

void CarControlDispatcher::onReceivedDirective(const std::string& action, const std::string& payload) {
    m_logger->log(Level::DEBUG, TAG, "Processing carcontrol directive: " + action);

    std::string vshlCapabilityAction = action;

    json_object* argsJ = json_object_new_object();
    json_object* actionJ = json_object_new_string(vshlCapabilityAction.c_str());
    json_object* payloadJ = NULL;

    if (payload.length()) {
        payloadJ = json_tokener_parse(payload.c_str());
    } else {
        m_logger->log(Level::ERROR, TAG, "Unable to parse payload JSON. Setting to empty string: " + payload);
        payloadJ = json_object_new_string("");
    }

    if (!payloadJ) {
        m_logger->log(Level::ERROR, TAG, "Unable to parse payload JSON: " + payload);
        return;
    }

    json_object_object_add(argsJ, agl::alexa::JSON_ATTR_ACTION.c_str(), actionJ);
    json_object_object_add(argsJ, agl::alexa::JSON_ATTR_PAYLOAD.c_str(), payloadJ);

    auto logger = m_logger;  // To capture it for the lambda.
    m_api->callAsync(
        agl::alexa::VSHL_CAPABILITIES_API_NAME,
        VSHL_CAPABILITY_VERB_CAR_CONTROL_PUBLISH,
        argsJ,
        [logger](json_object* result, const std::string err, const std::string info) {
            logger->log(Level::DEBUG, TAG, "Carcontrol publish completed err:" + err);
        });

    m_logger->log(Level::DEBUG, TAG, "Carcontrol action processing completed");
}

bool CarControlDispatcher::subscribeToCarControlEvents() {
    m_logger->log(Level::INFO, TAG, "Subscribing to carcontrol capabilities");

    json_object* argsJ = json_object_new_object();
    json_object* actionsJ = json_object_new_array();
    json_object_array_add(
        actionsJ, json_object_new_string(aasb::bridge::ACTION_CARCONTROL_IS_POWER_CONTROLLER_ON_RESPONSE.c_str()));
    json_object_array_add(
        actionsJ, json_object_new_string(aasb::bridge::ACTION_CARCONTROL_IS_TOGGLE_CONTROLLER_ON_RESPONSE.c_str()));
    json_object_array_add(
        actionsJ, json_object_new_string(aasb::bridge::ACTION_CARCONTROL_GET_MODE_CONTROLLER_VALUE_RESPONSE.c_str()));
    json_object_array_add(
        actionsJ, json_object_new_string(aasb::bridge::ACTION_CARCONTROL_GET_RANGE_CONTROLLER_VALUE_RESPONSE.c_str()));

    json_object_object_add(argsJ, agl::alexa::JSON_ATTR_ACTIONS.c_str(), actionsJ);

    json_object* response = NULL;
    std::string error, info;
    int result = m_api->callSync(
        agl::alexa::VSHL_CAPABILITIES_API_NAME,
        VSHL_CAPABILITY_VERB_CAR_CONTROL_SUBSCRIBE,
        argsJ,
        &response,
        error,
        info);
    if (result != 0) {
        m_logger->log(Level::ERROR, TAG, "Failed to subscribe to carcontrol control capabilities. Error: " + error);
    }

    if (response != NULL) {
        json_object_put(response);
    }

    return result == 0;
}

void CarControlDispatcher::onIsPowerControllerOnResponse(const std::string& payload) {
    m_aasbController->onReceivedEvent(TOPIC_CARCONTROL, ACTION_CARCONTROL_IS_POWER_CONTROLLER_ON_RESPONSE, payload);
}

void CarControlDispatcher::onIsToggleControllerOnResponse(const std::string& payload) {
    m_aasbController->onReceivedEvent(TOPIC_CARCONTROL, ACTION_CARCONTROL_IS_TOGGLE_CONTROLLER_ON_RESPONSE, payload);
}

void CarControlDispatcher::onGetModeControllerValueResponse(const std::string& payload) {
    m_aasbController->onReceivedEvent(TOPIC_CARCONTROL, ACTION_CARCONTROL_GET_MODE_CONTROLLER_VALUE_RESPONSE, payload);
}

void CarControlDispatcher::onGetRangeControllerValueResponse(const std::string& payload) {
    m_aasbController->onReceivedEvent(TOPIC_CARCONTROL, ACTION_CARCONTROL_GET_RANGE_CONTROLLER_VALUE_RESPONSE, payload);
}

}  // namespace carControl
}  // namespace localVoiceControl
}  // namespace dispatcher
}  // namespace agl
